/*
 * StraightMovementInfo.h
 *
 * Created 8/31/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/31/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Contains information for a relative straight movement.
  */
 
 #ifndef STRAIGHT_MOVEMENT_INFO_H
 #define STRAIGHT_MOVEMENT_INFO_H
 
 template <typename T> class StraightMovementInfo;
 
 #include "global.h"
 
 #include "MovementInfo.h"
 
 /**
  * Class specification for StraightMovementInfo
  */
 template <typename T>
 class StraightMovementInfo : public MovementInfo<T>
 {
 // Data Members
 private:
    VECTOR3_TYPE _direction; // geographical coordinates vector pointing in a direction (magnitude matters)
    
 // Local Functions
 public:
    StraightMovementInfo( const VECTOR3_TYPE& direction = VECTOR3_TYPE( ZERO, ZERO, ZERO ), 
                          const T distance = ZERO, const double duration = 0.0 );
    StraightMovementInfo( const StraightMovementInfo<T>& straight_movement_info );
    virtual ~StraightMovementInfo();
    inline StraightMovementInfo<T>& operator=( const StraightMovementInfo<T>& straight_movement_info );
    inline T get_distance() const;
    inline VECTOR3_TYPE get_direction() const;
    inline void set_distance( const T distance );
    inline void set_direction( const VECTOR3_TYPE& direction );
    virtual inline double process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 StraightMovementInfo<T>::StraightMovementInfo( const VECTOR3_TYPE& direction, const T distance, const double duration )
                         : MovementInfo<T>( duration ),
                           _direction( direction * distance )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 StraightMovementInfo<T>::StraightMovementInfo( const StraightMovementInfo<T>& straight_movement_info )
                         : MovementInfo<T>( straight_movement_info ),
                           _direction( straight_movement_info._direction )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 StraightMovementInfo<T>::~StraightMovementInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified StraightMovementInfo to this StraightMovementInfo.
  *
  * @param (const StraightMovementInfo<T>&) straight_movement_info
  * @return StraightMovementInfo<T>&
  */
 template <typename T>
 inline StraightMovementInfo<T>& StraightMovementInfo<T>::operator=( const StraightMovementInfo<T>& straight_movement_info )
 {
    MovementInfo<T>::operator=( straight_movement_info );
    _direction = straight_movement_info._direction;
    
    return *this;
 }
 
 /**
  * get_distance() returns the movement distance.
  *
  * @return T
  */
 template <typename T>
 inline T StraightMovementInfo<T>::get_distance() const
 {
    return Vector::get_magnitude( _direction );
 }
 
 /**
  * get_direction() returns the movement direction.
  *
  * @return VECTOR3_TYPE
  */
 template <typename T>
 inline VECTOR3_TYPE StraightMovementInfo<T>::get_direction() const
 {
    return _direction;
 }
 
 /**
  * set_distance() sets the movement distance to the specified distance.
  *
  * @param (const T) distance
  */
 template <typename T>
 inline void StraightMovementInfo<T>::set_distance( const T distance )
 {
    _direction = Vector::get_normal(_direction) * distance;
 }
 
 /**
  * set_direction() sets the movement direction to the specified direction.
  *
  * @param (const VECTOR3_TYPE&) direction
  */
 template <typename T>
 inline void StraightMovementInfo<T>::set_direction( const VECTOR3_TYPE& direction )
 {
    _direction = direction;
 }
 
 /**
  * process_movement() processes the movement specified by this MovementInfo, given the Object to process on,
  * the MoveInfo, and current time. The duration unprocessed returned is greater than zero if this MovementInfo
  * has finished processing, and there is time leftover.
  *
  * @param (Object<T>*) obj_Ptr
  * @param (MoveInfo<T>*) move_info_Ptr
  * @param (double) current_time
  * @return double - the time left unprocessed (i.e. a double greater than zero if this MovementInfo is done processing)
  */
 template <typename T>
 inline double StraightMovementInfo<T>::process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time )
 {  
    nassertr( obj_Ptr != NULL, MovementInfo<T>::get_duration() );
    
    double move_action_invoke_time( move_info_Ptr->get_time_move_action_was_invoked() );
    double last_process_time( move_info_Ptr->get_time_last_processed_move_task() );
    nassertr( last_process_time >= move_action_invoke_time, MovementInfo<T>::get_duration() );
    
    double duration( MovementInfo<T>::get_duration() );
    double elapse_time_since_invocation( current_time - move_action_invoke_time );
    
    // gradually move for only the movement duration (measured in seconds)
    double elapse_time_since_last_movement( current_time - last_process_time );
    if ( elapse_time_since_invocation > duration ) // if true, this will be the last time we process this movement_info
            elapse_time_since_last_movement = duration + move_action_invoke_time - last_process_time;
    
    if ( elapse_time_since_last_movement > 0.0 && StraightMovementInfo<T>::get_direction() != VECTOR3_TYPE(ZERO,ZERO,ZERO) )
    {
        double movement_ratio( elapse_time_since_last_movement / duration );
        
        // translate Object by specified distance units in the specified direction
        obj_Ptr->translate( StraightMovementInfo<T>::get_direction() * movement_ratio );
        // sometimes cause great floating-point error
    }
    
    if ( elapse_time_since_invocation > duration )
        return elapse_time_since_invocation - duration;
    else // elapse_time_since_invocation <= duration
        return 0.0;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // STRAIGHT_MOVEMENT_INFO_H